// $Id: CMusicMath.cpp,v 1.8 2007/03/07 22:19:19 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CMusicMath.hpp"
#include <MathTools/CBounds.hpp>
#include <Exceptions/CException.hpp>
#include <MathTools/CMathTools.hpp>

//	===========================================================================

using Exponent::Music::CMusicMath;
using Exponent::MathTools::CBounds;
using Exponent::Exceptions::CException;
using Exponent::MathTools::CMathTools;

//	===========================================================================
const CMusicMath::CFrequencyTable CMusicMath::CMUSIC_MATH_FREQUENCY_TABLE;

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_TRIPLET_TIME = 1.0 / 1.5;

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_DOTTED_TIME  = 5.0 / 7.5;

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_FREQUENCY[] =
{
	0.00390625,
	0.0078125,
	0.015625,
	0.03125,
	0.0625,
	0.125,
	0.25,
	0.5,
	1.0,
	2.0,
	4.0,
	8.0,
	16.0
};

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_TIME[] =
{
	0.015625,
	0.03125,
	0.0625,
	0.125,
	0.25,
	0.5,
	1.0,
	2.0,
	4.0,
	8.0,
	16.0,
	32.0,
	64.0
};

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_SONG_PHASE[] =
{
	0.015625,
	0.03125,
	0.0625,
	0.125,
	0.25,
	0.5,
	1.0,
	2.0,
	4.0,
	8.0,
	16.0,
	32.0,
	64.0
};

//	===========================================================================
const double CMusicMath::CMUSIC_MATH_SONG_POSITION[] =
{
	64.0,
	32.0,
	16.0,
	8.0,
	4.0,
	2.0,
	1.0,
	0.5,
	0.25,
	0.125,
	0.0625,
	0.03125,
	0.015625
};

//	===========================================================================
double CMusicMath::getRateInMilliseconds(const double bpm, const ETempoDivision tempoDivision, const CTimeSignature &timeSignature)
{
	// Type of division
	ETempoDivisionType type = e_straight;

	// Get the adjusted division
	const ETempoDivision division = CMusicMath::getAdjustedTempoDivision(tempoDivision, type);

	// Compute the lfo frequency
	return CMusicMath::convertValueToTempoDivisionType((((60.0 / (bpm / timeSignature.getDenominator())) * 1000.0) / CMUSIC_MATH_TIME[division]), type);
}

//	===========================================================================
double CMusicMath::getRateInHertz(const double bpm, const ETempoDivision tempoDivision)
{
	// Type of division
	ETempoDivisionType type = e_straight;

	// Get the adjusted division
	const ETempoDivision division = CMusicMath::getAdjustedTempoDivision(tempoDivision, type);

	// Compute the lfo frequency
	return CMusicMath::convertValueToTempoDivisionType((1000.0 / ((60000.0 / bpm)) * CMUSIC_MATH_FREQUENCY[division]), type);
}

//	===========================================================================
CMusicMath::ETempoDivision CMusicMath::getClosestTempoDivisionToRateInHertz(const double bpm, const double frequency)
{
	// First we convert it to the normalised value
	const double convertedValue = frequency / (1000.0 / (60000.0 / bpm));

	// Next we find the closest match to the normalised value
	for (long i = 1; i < e_sixtyFourBarsTriplet - 1; i++)
	{
		// Check the range of the converted value to find the closest value
		if (convertedValue >= CMUSIC_MATH_FREQUENCY[i] && convertedValue < CMUSIC_MATH_FREQUENCY[i + 1])
		{
			return (ETempoDivision)i;
		}
	}

	// Default back to quarter bar...
	return e_quarterBar;
}

//	===========================================================================
double CMusicMath::getPhase(const ETempoDivision tempoDivision)
{
	// Type of division
	ETempoDivisionType type = e_straight;

	// Get the adjusted division
	const ETempoDivision division = CMusicMath::getAdjustedTempoDivision(tempoDivision, type);

	// Return the frequency
	return CMusicMath::convertValueToTempoDivisionType(CMUSIC_MATH_SONG_PHASE[division], type);
}

//	===========================================================================
double CMusicMath::getSongPhasePosition(const ETempoDivision tempoDivision)
{
	// Type of division
	ETempoDivisionType type = e_straight;

	// Get the adjusted division
	const ETempoDivision division = CMusicMath::getAdjustedTempoDivision(tempoDivision, type);

	// Return the frequency
	return CMusicMath::convertValueToTempoDivisionType(CMUSIC_MATH_SONG_POSITION[division], type);
}

//	===========================================================================
long CMusicMath::getClosestMidiNote(const double frequency)
{
	return (long)(floor(69.5 + (12.0 * log(frequency / 440.0) / log(2.0))));
}

//	===========================================================================
double CMusicMath::getNoteFrequency(const long note)
{
	return 440.00 * pow(2.0, (double)((note - 69.0) / 12.0));
}

//	===========================================================================
double CMusicMath::getNoteFrequency(const long note, const long octaveDetune, const long semiDetune, const long fineDetune)
{
	// remove or add the octave detune...
	long adjustedNote = note + (octaveDetune * 12);
	adjustedNote += semiDetune;
	adjustedNote = CBounds::ensureRange(adjustedNote, 0, 127);

	// get our base frequency
	double frequency = getNoteFrequency(note);

	if (fineDetune > 0)				// +ve detune
	{
		frequency += ((CMUSIC_MATH_FREQUENCY_TABLE.getFrequency(CBounds::ensureRange(adjustedNote + 1, 0, 127)) - frequency) * 0.01) * fineDetune;
	}
	else if (fineDetune < 0)		// -ve detune
	{
		frequency += ((CMUSIC_MATH_FREQUENCY_TABLE.getFrequency(CBounds::ensureRange(adjustedNote - 1, 0, 127)) - frequency) * 0.01) * fineDetune;
	}

	return frequency;
}

//	===========================================================================
double CMusicMath::getNumberOfSamples(const double sampleRate, const double lengthInSeconds)
{
	return lengthInSeconds * sampleRate;
}

//	===========================================================================
CMusicMath::ETempoDivision CMusicMath::getAdjustedTempoDivision(const ETempoDivision division, ETempoDivisionType &type)
{
	if ((division >= e_sixtyFourBarsDotted  && division <= e_sixtyFourthBarDotted))
	{
		type = e_dotted;
		return (ETempoDivision)(division - e_sixtyFourBarsDotted);
	}
	else if ((division >= e_sixtyFourBarsTriplet && division <= e_sixtyFourthBarTriplet))
	{
		type = e_triplet;
		return (ETempoDivision)(division - e_sixtyFourBarsTriplet);
	}

	type = e_straight;
	return division;
}

//	===========================================================================
double CMusicMath::convertValueToTempoDivisionType(const double value, const ETempoDivisionType type)
{
	if (type == e_triplet)
	{
		return value * CMUSIC_MATH_TRIPLET_TIME;
	}
	else if (type == e_dotted)
	{
		return value * CMUSIC_MATH_DOTTED_TIME;
	}
	return value;
}

//	===========================================================================
double CMusicMath::getTempoNoteShift(const double originalBpm, const double newBpm)
{
	return log(newBpm / originalBpm) / 0.05776227;
}

//	===========================================================================
double CMusicMath::getTimeStretch(const double originalBpm, const double newBpm)
{
	return (originalBpm / newBpm) * 100.0;
}

//	===========================================================================
double CMusicMath::getNewTempoPitchShift(const double shift, const double bpm)
{
	return pow(1.0594631, shift) * bpm;
}

//	===========================================================================
double CMusicMath::getBpmFromTime(const double length, const CTimeSignature &timeSignature, const long numberOfBeats)
{
	return 60000.0 / (((length * 1000.0 / timeSignature.getAsDecimal()) / numberOfBeats) * 0.25);
}

//	===========================================================================
double CMusicMath::getTimeFromBpm(const double bpm, const CTimeSignature &timeSignature, const long numberOfBeats)
{
	return ((60000.0 / (bpm * 1000.0)) * 4.0 * numberOfBeats * timeSignature.getAsDecimal());
}

//	===========================================================================
void CMusicMath::getADivisionString(CString &theString, const ETempoDivision tempoDivision)
{
	switch(tempoDivision)
	{
		case e_sixtyFourBars:				theString = "64";			break;
		case e_thirtyTwoBars:				theString = "32";			break;
		case e_sixteenBars:					theString = "16";			break;
		case e_eightBars:					theString = "8";			break;
		case e_fourBars:					theString = "4";			break;
		case e_twoBars:						theString = "2";			break;
		case e_oneBar:						theString = "1";			break;
		case e_halfBar: 					theString = "1/2";			break;
		case e_quarterBar: 					theString = "1/4";			break;
		case e_eigthBar: 					theString = "1/8";			break;
		case e_sixteenthBar: 				theString = "1/16";			break;
		case e_thirtySecondBar:				theString = "1/32";			break;
		case e_sixtyFourthBar:				theString = "1/64";			break;
		case e_sixtyFourBarsTriplet:		theString = "64T";			break;
		case e_thirtyTwoBarsTriplet:		theString = "32T";			break;
		case e_sixteenBarsTriplet:			theString = "16T";			break;
		case e_eightBarsTriplet:			theString = "8T";			break;
		case e_fourBarsTriplet:				theString = "4T";			break;
		case e_twoBarsTriplet:				theString = "2T";			break;
		case e_oneBarTriplet:				theString = "1T";			break;
		case e_halfBarTriplet:				theString = "1/2T";			break;
		case e_quarterBarTriplet:			theString = "1/4T";			break;
		case e_eigthBarTriplet:				theString = "1/8T";			break;
		case e_sixteenthBarTriplet:			theString = "1/16T";		break;
		case e_thirtySecondBarTriplet:		theString = "1/32T";		break;
		case e_sixtyFourthBarTriplet:		theString = "1/64T";		break;
		case e_sixtyFourBarsDotted:			theString = "64D";			break;
		case e_thirtyTwoBarsDotted:			theString = "32D";			break;
		case e_sixteenBarsDotted:			theString = "16D";			break;
		case e_eightBarsDotted:				theString = "8D";			break;
		case e_fourBarsDotted:				theString = "4D";			break;
		case e_twoBarsDotted: 				theString = "2D";			break;
		case e_oneBarDotted:				theString = "1D";			break;
		case e_halfBarDotted: 				theString = "1/2D";			break;
		case e_quarterBarDotted: 			theString = "1/4D";			break;
		case e_eigthBarDotted: 				theString = "1/8D";			break;
		case e_sixteenthBarDotted: 			theString = "1/16D";		break;
		case e_thirtySecondBarDotted: 		theString = "1/32D";		break;
		case e_sixtyFourthBarDotted:		theString = "1/64D";		break;
	}
}